import {
    createProduct,
    getProducts,
    getProductId,
    updateProduct,
    deleteProduct,
    getInfoLibrary,
    saveInfoLibrary
} from "../services/productService.js";

/* ---------------------- Utility ---------------------- */
function parseBody(req) {
    try {
        return req.body.data ? JSON.parse(req.body.data) : req.body;
    } catch {
        return req.body;
    }
}

/* ---------------------- CREATE ---------------------- */
export const createProductController = async (req, res) => {
    try {
        const data = parseBody(req);
        const product = await createProduct(data, req.files);
        res.status(201).json(product);
    } catch (err) {
        console.error("Error creating product:", err);
        res.status(400).json({ message: err.message });
    }
};

/* ---------------------- UPDATE ---------------------- */
export const updateProductController = async (req, res) => {
    try {
        const data = parseBody(req);
        const updated = await updateProduct(req.params.id, data, req.files);
        res.json(updated);
    } catch (err) {
        console.error("Error updating product:", err);
        res.status(400).json({ message: err.message });
    }
};

/* ---------------------- GET ALL PRODUCTS ---------------------- */
export const getAllProductsController = async (req, res) => {
    try {
        const result = await getProducts(req.query);
        res.json(result);
    } catch (err) {
        console.error("Error fetching products:", err);
        res.status(400).json({ message: err.message });
    }
};

/* ---------------------- GET ONE PRODUCT ---------------------- */
export const getProductIdController = async (req, res) => {
    try {
        const product = await getProductId(req.params.id);
        if (!product) return res.status(404).json({ message: "Product not found" });
        res.json(product);
    } catch (err) {
        console.error("Error fetching product:", err);
        res.status(404).json({ message: err.message });
    }
};

/* ---------------------- DELETE ---------------------- */
export const deleteProductController = async (req, res) => {
    try {
        const result = await deleteProduct(req.params.id);
        res.json(result);
    } catch (err) {
        console.error("Error deleting product:", err);
        res.status(400).json({ message: err.message });
    }
};

/* ---------------------- INFO LIBRARY ---------------------- */
export const getInfoLibraryController = async (req, res) => {
    try {
        const library = await getInfoLibrary();
        res.json(library);
    } catch (err) {
        console.error("Error fetching info library:", err);
        res.status(500).json({ message: err.message });
    }
};

export const saveInfoLibraryController = async (req, res) => {
    try {
        const { sections } = req.body;
        const saved = await saveInfoLibrary(sections);
        res.json(saved);
    } catch (err) {
        console.error("Error saving info library:", err);
        res.status(500).json({ message: err.message });
    }
};
