import mongoose from "mongoose";

/* --------------------------------------------------
    GLOBAL INFO SECTION LIBRARY MODEL
-------------------------------------------------- */
const InfoSectionLibrarySchema = new mongoose.Schema(
    {
        _id: { type: String, default: "GLOBAL_INFO_LIBRARY" },
        sections: [
            {
                id: String,
                title: String,
                html: String,
            },
        ],
    },
    { timestamps: true }
);

export const InfoSectionLibrary = mongoose.model(
    "InfoSectionLibrary",
    InfoSectionLibrarySchema
);

/* --------------------------------------------------
    PRODUCT MODEL
-------------------------------------------------- */
const productSchema = new mongoose.Schema(
    {
        /* ---------- BASIC DETAILS ---------- */
        name: { type: String, required: true, trim: true },
        ribbon: { type: String }, // Example: "New", "Trending", "Sale"
        brand: { type: String, required: true },
        description: { type: String, required: true },
        rating: {
            type: Number,
            default: 0,
        },
        reviewsCount: {
            type: Number,
            default: 0,
        },

        /* ---------- CATEGORY ---------- */
        category: {
            type: mongoose.Schema.Types.ObjectId,
            ref: "Category",
            required: true,
        },

        /* ---------- PRICING ---------- */
        price: { type: Number, required: true },
        strikethroughPrice: { type: Number },
        costOfGoods: { type: Number },
        profit: { type: Number },
        margin: { type: Number },

        /* ---------- MEDIA ---------- */
        media: [
            {
                url: { type: String },
                type: {
                    type: String,
                    enum: ["image", "video"],
                    default: "image",
                },
            },
        ],

        /* ---------- INVENTORY ---------- */
        trackInventory: { type: Boolean, default: false },
        stockQuantity: { type: Number, default: 0 },
        sku: { type: String },
        barcode: { type: String },
        status: {
            type: String,
            enum: ["in_stock", "out_of_stock", "pre_order"],
            default: "in_stock",
        },

        /* ---------- PRODUCT OPTIONS (COLOR, SIZE, ETC.) ---------- */
        options: [
            {
                name: String, // e.g. "Color" or "Size"
                type: {
                    type: String,
                    enum: ["dropdown", "color"],
                    default: "dropdown",
                },
                values: [String], // e.g. ["Red", "Blue", "Green"]
                images: [String], // e.g. ["/uploads/red.jpg", "/uploads/blue.jpg"]
            },
        ],

        /* ---------- MODIFIERS (CUSTOMIZATIONS) ---------- */
        modifiers: [
            {
                id: String,
                name: String,
                type: { type: String, enum: ["dropdown", "color", "text"] },
                required: Boolean,
                choices: [String],
                characterLimit: Number,
                order: Number,
            },
        ],

        /* ---------- VARIANTS ---------- */
        variants: [
            {
                optionCombination: { type: Map, of: String }, // e.g. { Color: "Red", Size: "M" }
                price: Number,
                strikethroughPrice: Number,
                costOfGoods: Number,
                trackQuantity: Boolean,
                stockQuantity: Number,
                allowPreorder: Boolean,
                preorderLimit: Number,
                sku: String,
                barcode: String,

                // ✅ each variant can have its own image(s)
                media: [
                    {
                        url: String,
                        type: { type: String, enum: ["image", "video"], default: "image" },
                    },
                ],

                shippingEnabled: Boolean,
                shippingWeight: Number,
                shippingUnit: {
                    type: String,
                    enum: ["g", "kg", "ml", "L", "lb", "oz"],
                    default: "g",
                },
            },
        ],

        /* ---------- ADDITIONAL INFO SECTIONS ---------- */
        infoSections: [
            {
                id: String,
                title: String,
                html: String,
                customTitle: String,
                customHtml: String,
                order: Number,
            },
        ],
    },
    { timestamps: true }
);

export default mongoose.model("Product", productSchema);
