import { useState, useEffect } from "react";
import { Link, useLocation } from "react-router-dom";
import { FontAwesomeIcon } from "@fortawesome/react-fontawesome";
import {
  faTachometerAlt,
  faBoxOpen,
  faShoppingCart,
  faUsers,
  faStore,
  faChevronDown,
  faChevronRight,
} from "@fortawesome/free-solid-svg-icons";

export default function AdminSidebar() {
  const location = useLocation();
  const [openMenu, setOpenMenu] = useState(null);

  useEffect(() => {
    if (
      location.pathname.startsWith("/admin/products") ||
      location.pathname.startsWith("/admin/inventory") ||
      location.pathname.startsWith("/admin/categories") ||
      location.pathname.startsWith("/admin/back-in-stock") ||
      location.pathname.startsWith("/admin/dropshipping")
    ) {
      setOpenMenu("Catalog");
    }
  }, [location.pathname]);

    const menuItems = [
        { name: "Dashboard", path: "/admin/dashboard", icon: faTachometerAlt },
        {
            name: "Catalog",
            icon: faBoxOpen,
            subItems: [
                { name: "Products", path: "/admin/products" },
                { name: "Inventory", path: "/admin/inventory" },
                { name: "Categories", path: "/admin/category" },
                { name: "Back In Stock", path: "/admin/back-in-stock" },
                { name: "Dropshipping", path: "/admin/dropshipping" },
            ],
        },
        { name: "Orders", path: "/admin/orders", icon: faShoppingCart },
        { name: "Users", path: "/admin/users", icon: faUsers },
        { name: "Vendors", path: "/admin/vendors", icon: faStore },
    ];

  const toggleMenu = (name) => {
    setOpenMenu(openMenu === name ? null : name);
  };

  return (
    <aside className="sticky top-0 left-0 h-screen w-64 bg-white shadow-lg flex flex-col">
      {/* Sidebar Content */}
      <nav className="flex-1 overflow-y-auto p-4">
        <ul className="space-y-2">
          {menuItems.map((item) => {
            const isActive =
              location.pathname === item.path ||
              item.subItems?.some((sub) => location.pathname === sub.path);

            return (
              <li key={item.name}>
                {item.subItems ? (
                  <div
                    onClick={() => toggleMenu(item.name)}
                    className={`flex items-center justify-between gap-3 px-4 py-3 rounded-lg cursor-pointer transition-all duration-200 ${
                      isActive
                        ? "bg-orange-500 text-white shadow-md"
                        : "text-gray-700 hover:bg-orange-50 hover:text-orange-600"
                    }`}
                  >
                    <div className="flex items-center gap-3">
                      <FontAwesomeIcon icon={item.icon} className="w-5 h-5" />
                      <span className="font-medium">{item.name}</span>
                    </div>
                    <FontAwesomeIcon
                      icon={
                        openMenu === item.name ? faChevronDown : faChevronRight
                      }
                      className="w-4 h-4"
                    />
                  </div>
                ) : (
                  <Link
                    to={item.path}
                    className={`flex items-center gap-3 px-4 py-3 rounded-lg transition-all duration-200 ${
                      isActive
                        ? "bg-orange-500 text-white shadow-md"
                        : "text-gray-700 hover:bg-orange-50 hover:text-orange-600"
                    }`}
                  >
                    <FontAwesomeIcon icon={item.icon} className="w-5 h-5" />
                    <span className="font-medium">{item.name}</span>
                  </Link>
                )}

                {item.subItems && openMenu === item.name && (
                  <ul className="ml-8 mt-2 space-y-2 border-l-2 border-orange-100 pl-3">
                    {item.subItems.map((sub) => {
                      const isSubActive = location.pathname === sub.path;
                      return (
                        <li key={sub.path}>
                          <Link
                            to={sub.path}
                            className={`block px-3 py-2 rounded-md text-sm transition-all duration-150 ${
                              isSubActive
                                ? "bg-orange-100 text-orange-600 font-semibold"
                                : "text-gray-600 hover:bg-orange-50 hover:text-orange-600"
                            }`}
                          >
                            {sub.name}
                          </Link>
                        </li>
                      );
                    })}
                  </ul>
                )}
              </li>
            );
          })}
        </ul>
      </nav>
    </aside>
  );
}
