import React, { useState } from "react";
import { X, Info, Tag, Percent, Truck, DollarSign, Gift } from "lucide-react";

export default function NewCouponModal({ onClose }) {
  const [couponType, setCouponType] = useState("");
  const [couponCode, setCouponCode] = useState("");
  const [couponName, setCouponName] = useState("");
  const [discount, setDiscount] = useState("");
  const [discountPercent, setDiscountPercent] = useState("");
  const [salePrice, setSalePrice] = useState("");
  const [buyQty, setBuyQty] = useState("");
  const [getQty, setGetQty] = useState("");
  const [applyTo, setApplyTo] = useState("Specific product");
  const [product, setProduct] = useState("");
  const [includeSubs, setIncludeSubs] = useState(false);
  const [noEndDate, setNoEndDate] = useState(true);
  const [startDate, setStartDate] = useState("");
  const [endDate, setEndDate] = useState("");
  const [limitTotal, setLimitTotal] = useState(false);
  const [limitCustomer, setLimitCustomer] = useState(false);

  const inputClass =
    "w-full border border-gray-300 rounded-md p-2 focus:ring-2 focus:ring-orange-400 outline-none transition";

  const couponTypes = [
    {
      label: "₹ Discount",
      bgColor: "bg-red-500",
      activeColor: "bg-red-600",
      ring: "red-300",
      icon: <Tag />,
    },
    {
      label: "% Discount",
      bgColor: "bg-blue-500",
      activeColor: "bg-blue-600",
      ring: "blue-300",
      icon: <Percent />,
    },
    {
      label: "Free shipping",
      bgColor: "bg-yellow-400",
      activeColor: "bg-yellow-500",
      ring: "yellow-200",
      icon: <Truck />,
    },
    {
      label: "Sale price",
      bgColor: "bg-purple-500",
      activeColor: "bg-purple-600",
      ring: "purple-300",
      icon: <DollarSign />,
    },
    {
      label: "Buy X get Y free",
      bgColor: "bg-green-500",
      activeColor: "bg-green-600",
      ring: "green-300",
      icon: <Gift />,
    },
  ];

  const handleCreate = () => {
    const data = {
      couponType,
      couponCode,
      couponName,
      discount,
      discountPercent,
      salePrice,
      buyQty,
      getQty,
      applyTo,
      product,
      includeSubs,
      startDate,
      endDate,
      noEndDate,
      limitTotal,
      limitCustomer,
    };
    console.log("Created Coupon:", data);
    alert("Coupon Created! Check console for details.");
  };

  return (
    <div className="fixed inset-0 flex items-center justify-center bg-black/50 z-50 transition-all">
      <div className="bg-white rounded-2xl shadow-2xl w-full max-w-4xl p-10 relative overflow-y-auto max-h-[90vh] hide-scrollbar">
        {/* Header */}
        <div className="flex justify-between items-center border-b pb-4 mb-8">
          <h2 className="text-3xl font-semibold text-orange-600">New Coupon</h2>
          <button
            onClick={onClose}
            className="p-2 rounded-full hover:bg-orange-100 transition"
          >
            <X className="w-6 h-6 text-orange-600" />
          </button>
        </div>

        {/* Coupon Type */}
        <p className="mb-3 font-medium text-gray-700 text-lg">
          Choose your coupon type:
        </p>

        <div className="flex justify-between gap-3 mb-8">
          {couponTypes.map((type) => (
            <button
              key={type.label}
              onClick={() => setCouponType(type.label)}
              className={`flex flex-col items-center justify-center flex-1 py-5 rounded-xl font-semibold text-sm transition-all duration-200 shadow-md hover:scale-105 ${
                couponType === type.label
                  ? `${type.activeColor} text-white shadow-lg ring-4 ring-${type.ring}`
                  : `${type.bgColor} text-white opacity-90 hover:opacity-100`
              }`}
            >
              <div className="mb-2 text-lg">{type.icon}</div>
              {type.label}
            </button>
          ))}
        </div>

        {/* Coupon Details */}
        <div className="grid grid-cols-1 md:grid-cols-2 gap-6 mb-8">
          <div>
            <label className="font-medium flex items-center gap-1 text-gray-700">
              Coupon code <Info size={16} className="text-orange-400" />
            </label>
            <input
              type="text"
              placeholder="e.g., SUMMERSALE20"
              className={inputClass}
              value={couponCode}
              onChange={(e) => setCouponCode(e.target.value)}
            />
          </div>
          <div>
            <label className="font-medium text-gray-700">Coupon name</label>
            <input
              type="text"
              placeholder="e.g., Summer sale"
              className={inputClass}
              value={couponName}
              onChange={(e) => setCouponName(e.target.value)}
            />
          </div>
        </div>

        {/* Dynamic Fields */}
        {couponType === "₹ Discount" && (
          <div className="mb-6">
            <label className="font-medium text-gray-700">
              Discount Amount (₹)
            </label>
            <input
              type="number"
              placeholder="Enter discount amount"
              className={inputClass}
              value={discount}
              onChange={(e) => setDiscount(e.target.value)}
            />
          </div>
        )}

        {couponType === "% Discount" && (
          <div className="mb-6">
            <label className="font-medium text-gray-700">
              Discount Percentage (%)
            </label>
            <input
              type="number"
              placeholder="Enter discount percent"
              className={inputClass}
              value={discountPercent}
              onChange={(e) => setDiscountPercent(e.target.value)}
            />
          </div>
        )}

        {couponType === "Sale price" && (
          <div className="mb-6">
            <label className="font-medium text-gray-700">Set Sale Price (₹)</label>
            <input
              type="number"
              placeholder="Enter sale price"
              className={inputClass}
              value={salePrice}
              onChange={(e) => setSalePrice(e.target.value)}
            />
          </div>
        )}

        {couponType === "Buy X get Y free" && (
          <div className="mb-6 space-y-3">
            <div className="flex items-center gap-3">
              <label className="text-sm text-gray-700">Buy</label>
              <input
                type="number"
                min="1"
                value={buyQty}
                onChange={(e) => setBuyQty(e.target.value)}
                className="w-20 border border-gray-300 rounded-md p-2 text-center focus:ring-2 focus:ring-orange-400 outline-none"
              />
              <label className="text-sm text-gray-700">Get</label>
              <input
                type="number"
                min="1"
                value={getQty}
                onChange={(e) => setGetQty(e.target.value)}
                className="w-20 border border-gray-300 rounded-md p-2 text-center focus:ring-2 focus:ring-orange-400 outline-none"
              />
              <span className="text-sm text-gray-700">Free</span>
            </div>
            <p className="text-xs text-gray-500">
              e.g., Buy 2 apples, get 1 for free.
            </p>
          </div>
        )}

        {couponType === "Free shipping" && (
          <div className="mb-6 text-sm text-gray-700 bg-orange-50 p-4 rounded-xl border border-orange-200">
            <p>
              This coupon provides <b>free shipping</b> on eligible orders.
            </p>
          </div>
        )}

        {/* Apply To */}
        {couponType && (
          <div className="mb-6">
            <label className="font-medium text-gray-700">Apply to</label>
            <select
              className={`${inputClass} cursor-pointer`}
              value={applyTo}
              onChange={(e) => setApplyTo(e.target.value)}
            >
              <option>Specific product</option>
              <option>All products</option>
            </select>
          </div>
        )}

        {/* Product */}
        {applyTo === "Specific product" && (
          <div className="mb-6">
            <label className="font-medium text-gray-700">Product</label>
            <input
              type="text"
              placeholder="Enter Product ID or Name"
              className={inputClass}
              value={product}
              onChange={(e) => setProduct(e.target.value)}
            />
            <p className="text-sm text-gray-500 mt-1">
              Current price: ₹100 (example)
            </p>
          </div>
        )}

        {/* Include Subscriptions */}
        <div className="flex items-center mb-4">
          <input
            type="checkbox"
            className="mr-2 accent-orange-500"
            checked={includeSubs}
            onChange={(e) => setIncludeSubs(e.target.checked)}
          />
          <label className="text-gray-700 text-sm">
            Include subscriptions – allow for recurring purchases.
          </label>
        </div>

        {/* Validity */}
        <div className="mb-8">
          <label className="font-medium flex items-center gap-1 text-gray-700">
            Valid between <Info size={16} className="text-orange-400" />
          </label>
          <div className="flex flex-wrap items-center gap-3 mt-2">
            <input
              type="date"
              className="border border-gray-300 rounded-md p-2 focus:ring-2 focus:ring-orange-400 outline-none"
              value={startDate}
              onChange={(e) => setStartDate(e.target.value)}
            />
            <input
              type="date"
              className="border border-gray-300 rounded-md p-2 focus:ring-2 focus:ring-orange-400 outline-none disabled:bg-gray-100"
              value={endDate}
              disabled={noEndDate}
              onChange={(e) => setEndDate(e.target.value)}
            />
            <label className="flex items-center gap-2 text-sm text-gray-700 ml-auto">
              <input
                type="checkbox"
                checked={noEndDate}
                onChange={(e) => setNoEndDate(e.target.checked)}
                className="accent-orange-500 h-4 w-4"
              />
              Don't set an end date
            </label>
          </div>
        </div>

        {/* Usage Limits */}
        <div className="mb-6">
          <label className="font-medium flex items-center gap-1 text-gray-700">
            Limit uses <Info size={16} className="text-orange-400" />
          </label>
          <div className="flex flex-col gap-3 mt-3 text-sm text-gray-700">
            <label className="flex items-center gap-2">
              <input
                type="checkbox"
                checked={limitTotal}
                onChange={(e) => setLimitTotal(e.target.checked)}
                className="accent-orange-500 h-4 w-4"
              />
              Limit the total number of uses for this coupon
            </label>
            <label className="flex items-center gap-2">
              <input
                type="checkbox"
                checked={limitCustomer}
                onChange={(e) => setLimitCustomer(e.target.checked)}
                className="accent-orange-500 h-4 w-4"
              />
              Limit to one use per customer
            </label>
          </div>
        </div>

        {/* Buttons */}
        <div className="flex justify-end gap-4 border-t pt-6 mt-8">
          <button
            onClick={onClose}
            className="px-6 py-3 rounded-md border border-gray-300 text-gray-700 hover:bg-orange-50 transition"
          >
            Cancel
          </button>
          <button
            onClick={handleCreate}
            className="px-6 py-3 rounded-md bg-orange-600 text-white font-semibold hover:bg-orange-700 transition shadow-md"
          >
            Create Coupon
          </button>
        </div>
      </div>
    </div>
  );
}

/* hide scrollbar globally */
<style jsx>{`
  .hide-scrollbar::-webkit-scrollbar {
    display: none;
  }
  .hide-scrollbar {
    -ms-overflow-style: none;
    scrollbar-width: none;
  }
`}</style>



