import { Info } from "lucide-react";
import { useState } from "react";

export default function Inventory({
    trackInventory,
    setTrackInventory,
    quantity,
    setQuantity,
    stockStatus,
    setStockStatus,
    sku,
    setSku,
    barcode,
    setBarcode,
    allowPreorder,
    setAllowPreorder,
    limitPreorder,
    setLimitPreorder,
    preorderLimit,
    setPreorderLimit,
    preorderMessage,
    setPreorderMessage,
}) {
    /* ------------------ Toggle UI ------------------ */
    const Toggle = ({ enabled, onToggle }) => (
        <button
            type="button"
            onClick={onToggle}
            className={`relative w-12 h-6 rounded-full transition-all flex items-center 
                ${enabled ? "bg-orange-500" : "bg-gray-300"}`}
        >
            <span
                className={`absolute w-5 h-5 bg-white rounded-full shadow transform transition-all
                    ${enabled ? "translate-x-6" : "translate-x-1"}`}
            />
        </button>
    );

    /* ------------------ Tooltip ------------------ */
    const Tooltip = ({ text }) => (
        <span className="relative group ml-1 cursor-pointer">
            <Info size={15} className="text-orange-500" />
            <span className="absolute left-1/2 -translate-x-1/2 bottom-full mb-2 hidden group-hover:block bg-black text-white text-xs px-2 py-1 rounded-md whitespace-nowrap shadow-lg z-50">
                {text}
            </span>
        </span>
    );

    return (
        <section className="bg-gray-50 border border-gray-200 rounded-xl p-6">
            <h2 className="text-2xl font-semibold mb-1">Inventory and pre-order</h2>
            <p className="text-gray-500 mb-4">
                Manage stock levels and pre-order settings for this product.
            </p>

            <hr className="mb-6" />

            {/* ✅ Track Inventory */}
            <div className="flex justify-between items-center mb-6">
                <span className="font-medium flex items-center gap-1">
                    Track inventory quantity <Tooltip text="Automatically track stock" />
                </span>
                <Toggle
                    enabled={trackInventory}
                    onToggle={() => setTrackInventory(!trackInventory)}
                />
            </div>

            {/* ✅ Quantity OR Stock Status */}
            {trackInventory ? (
                <div className="mb-6">
                    <label className="font-medium text-gray-700">Quantity</label>
                    <input
                        type="number"
                        value={quantity}
                        onChange={(e) => setQuantity(e.target.value)}
                        className="w-full mt-1 border rounded-lg px-3 py-2 bg-white focus:ring-2 focus:ring-blue-400"
                        placeholder="Enter available stock"
                    />
                </div>
            ) : (
                <div className="mb-6">
                    <label className="font-medium text-gray-700">Status</label>
                    <select
                        value={stockStatus}
                        onChange={(e) => setStockStatus(e.target.value)}
                        className="w-full mt-1 border rounded-lg px-3 py-2 bg-white focus:ring-2 focus:ring-blue-400"
                    >
                        <option value="in_stock">In stock</option>
                        <option value="out_of_stock">Out of stock</option>
                        <option value="preorder">Pre-order</option>
                    </select>
                </div>
            )}

            {/* ✅ SKU and Barcode */}
            <div className="grid grid-cols-1 md:grid-cols-2 gap-6 mb-6">
                <div>
                    <label className="font-medium flex items-center text-gray-700">
                        SKU <Tooltip text="Stock keeping unit identifier" />
                    </label>
                    <input
                        value={sku}
                        onChange={(e) => setSku(e.target.value)}
                        className="w-full border rounded-lg px-3 py-2 mt-1 bg-white focus:ring-2 focus:ring-blue-400"
                        placeholder="SKU"
                    />
                </div>

                <div>
                    <label className="font-medium flex items-center text-gray-700">
                        Barcode (GTIN) <Tooltip text="Global trade item number" />
                    </label>
                    <input
                        value={barcode}
                        onChange={(e) => setBarcode(e.target.value)}
                        className="w-full border rounded-lg px-3 py-2 mt-1 bg-white focus:ring-2 focus:ring-blue-400"
                        placeholder="Barcode"
                    />
                </div>
            </div>

            <hr className="mb-6" />

            {/* ✅ Pre-order Toggle */}
            <div className="flex justify-between items-center mb-3">
                <span className="font-medium flex items-center gap-1">
                    Sell as pre-order when this product is out of stock{" "}
                    <Tooltip text="Customers can purchase even when out of stock" />
                </span>
                <Toggle
                    enabled={allowPreorder}
                    onToggle={() => setAllowPreorder(!allowPreorder)}
                />
            </div>

            {allowPreorder && (
                <div className="mb-6">
                    <div className="bg-gray-100 border rounded-lg px-4 py-3 text-sm text-gray-700 mb-3">
                        This product is currently in stock. It will be sold as pre-order if
                        you mark it as out of stock.
                    </div>

                    {/* ✅ Preorder Message */}
                    <label className="font-medium flex items-center text-gray-800 mb-1">
                        Pre-order message <Tooltip text="Shown to customers before checkout" />
                    </label>
                    <textarea
                        rows="3"
                        value={preorderMessage}
                        onChange={(e) => setPreorderMessage(e.target.value)}
                        placeholder="e.g., Expected to ship by the end of June"
                        className="w-full border rounded-lg px-3 py-2 bg-white focus:ring-2 focus:ring-blue-400 mb-4"
                    />

                    {/* ✅ Limit Pre-orders */}
                    <div className="flex items-center gap-3 mb-4">
                        <input
                            type="checkbox"
                            checked={limitPreorder}
                            onChange={() => setLimitPreorder(!limitPreorder)}
                            className="w-4 h-4"
                        />
                        <span className="font-medium text-gray-700">
                            Limit the number of items available for pre-order
                        </span>
                    </div>

                    {/* ✅ Preorder Limit Box */}
                    {limitPreorder && (
                        <input
                            type="number"
                            value={preorderLimit}
                            onChange={(e) => setPreorderLimit(e.target.value)}
                            className="border rounded-lg px-3 py-2 w-40 bg-white focus:ring-2 focus:ring-blue-400"
                            placeholder="Enter limit"
                        />
                    )}
                </div>
            )}

            <hr className="my-6" />

            {/* ✅ Note Section */}
            <p className="text-sm text-gray-500 leading-relaxed">
                Note: Pre-orders can only be delivered within 12 months of the purchase
                date and are not part of a crowdfunding campaign.
            </p>
        </section>
    );
}
