import { useRef, useEffect, useState } from "react";
import { motion } from "framer-motion";
import { useNavigate } from "react-router-dom";
import agroImg from "../../assets/images/agro.png";
import custImg from "../../assets/images/cust.png";
import SellerSlide from "./SellerSlide";
import UniversalNavbar from "./UniversalNavbar";

const sections = [
  {
    id: "create",
    label: "Create Account",
    content: (
      <>
        <p>Registering as a seller on our platform is quick and simple.</p>
        <p>
          You’ll just need a few basic documents to get started. Once approved,
          you can start listing your products and selling online with ease.
        </p>
      </>
    ),
  },
  {
    id: "list",
    label: "List Products",
    content: (
      <div className="flex items-start gap-5 flex-col md:flex-row">
        <div className="flex-1">
          <p>
            A product listing refers to the process of adding your item to the
            platform, making it visible to customers so they can explore, view,
            and purchase it. Each listing creates a detailed product page that
            includes key information such as the product title, description,
            images, price, and specifications.
          </p>
          <p className="mt-2">
            A well-crafted listing helps your products stand out and boosts
            sales.
          </p>
        </div>
        <div className="flex-1">
          <img
            src={agroImg}
            alt="Product example"
            className="w-full max-w-md rounded-lg shadow-md"
          />
        </div>
      </div>
    ),
  },
  {
    id: "storage",
    label: "Storage & Shipping",
    content: (
      <>
        <p>
          Congratulations on receiving your first order! Fast, reliable, and
          secure delivery ensures customer satisfaction. Once confirmed, our
          logistics network ensures timely pickup and delivery while you manage
          product packing.
        </p>
        <p className="mt-2">
          <strong>Fulfilment by Seller (FBS)</strong> means you handle storage &
          packing, while we manage shipping.
        </p>
        <ul className="list-disc ml-6 mt-2">
          <li>Properly packed and labeled before pickup</li>
          <li>Real-time tracking for every order</li>
          <li>Customer returns handled</li>
        </ul>
      </>
    ),
  },
  {
    id: "payments",
    label: "Receive Payments",
    content: (
      <>
        <p>
          Once an order is delivered, your payment (after fees) is securely
          transferred to your bank account. Track earnings in your Dashboard.
        </p>
        <p className="mt-2">
          You can track earnings, completed transactions, and pending
          settlements easily. All payments are processed safely and
          transparently.
        </p>
      </>
    ),
  },
  {
    id: "support",
    label: "Help & Support",
    content: (
      <div className="flex items-start gap-5 flex-col md:flex-row">
        <div className="flex-1">
          <p>
            Our dedicated support team assists you promptly with any questions,
            concerns, or guidance related to your selling journey. Whether
            onboarding, managing products, or handling orders, we’re here to
            ensure a smooth and successful experience.
          </p>
        </div>
        <div className="flex-1">
          <img
            src={custImg}
            alt="Support"
            className="w-full max-w-md rounded-lg shadow-md"
          />
        </div>
      </div>
    ),
  },
];

const BecomeASeller = () => {
  const sectionRefs = useRef({});
  const [activeSection, setActiveSection] = useState("create");
  const navigate = useNavigate();

  const scrollToSection = (id) => {
    sectionRefs.current[id]?.scrollIntoView({
      behavior: "smooth",
      block: "start",
    });
  };

  useEffect(() => {
    const handleScroll = () => {
      const positions = Object.keys(sectionRefs.current).map((id) => ({
        id,
        top: Math.abs(sectionRefs.current[id].getBoundingClientRect().top),
      }));

      positions.sort((a, b) => a.top - b.top);
      if (positions[0] && positions[0].id !== activeSection) {
        setActiveSection(positions[0].id);
      }
    };

    window.addEventListener("scroll", handleScroll);
    return () => window.removeEventListener("scroll", handleScroll);
  }, [activeSection]);

  return (
    <div className="flex flex-col min-h-screen bg-gray-50 font-sans">
      {/* Universal Navbar */}
      <UniversalNavbar />

      {/* Seller Slide */}
      <div className="w-full bg-white shadow p-5 mb-6">
        <SellerSlide />
      </div>

      {/* Main Layout */}
      <div className="flex flex-1 px-5 md:px-12">
        {/* Sidebar */}
        <aside className="hidden md:flex flex-col w-64 space-y-2 sticky top-20 h-[calc(100vh-5rem)]">
          {sections.map((section) => (
            <button
              key={section.id}
              onClick={() => scrollToSection(section.id)}
              className={`w-full text-left px-3 py-2 rounded-lg transition-all duration-300 ${
                activeSection === section.id
                  ? "bg-orange-100 text-orange-600 font-semibold shadow-md scale-105"
                  : "hover:bg-gray-100 text-gray-700"
              }`}
            >
              {section.label}
            </button>
          ))}
        </aside>

        {/* Content */}
        <main className="flex-1 p-8 space-y-24">
          {sections.map((section) => (
            <motion.section
              key={section.id}
              id={section.id}
              ref={(el) => (sectionRefs.current[section.id] = el)}
              className="mb-24 scroll-mt-28"
              initial={{ opacity: 0, y: 40 }}
              whileInView={{ opacity: 1, y: 0 }}
              viewport={{ once: false, amount: 0.2 }}
              transition={{ duration: 0.6 }}
            >
              <h1 className="text-2xl font-bold mb-3 text-gray-900">
                {section.label}
              </h1>
              <div className="text-gray-700 text-base leading-relaxed">
                {section.content}
              </div>
            </motion.section>
          ))}
        </main>
      </div>
    </div>
  );
};

export default BecomeASeller;
