// Grow.jsx
import  { useRef, useEffect, useState } from "react";
import { useLocation } from "react-router-dom";
import { motion } from "framer-motion";
import SellerSlide from "./SellerSlide";
import UniversalNavbar from "./UniversalNavbar";

// Images
import ImgInsights from "../../assets/images/ImgInsights.png";
import ImgAds from "../../assets/images/ImgAds.png";
import ImgFestivals from "../../assets/images/ImgFestivals.png";

const sections = [
  {
    id: "insights",
    label: "Insights & Tools",
    content: (
      <div className="flex flex-col md:flex-row items-center gap-8 p-6">
        <div className="flex-1 space-y-3">
          <p>Powerful seller insights and tools to make smarter business decisions.</p>
          <ul className="list-disc ml-6">
            <li>Price Recommendations Tool</li>
            <li>Selection Insights</li>
            <li>Inventory Health</li>
          </ul>
        </div>
        <div className="flex-1 flex justify-center">
          <img
            src={ImgInsights}
            alt="Insights"
            className="w-2/3 max-w-xs rounded-lg shadow-md object-contain"
          />
        </div>
      </div>
    ),
  },
  {
    id: "ads",
    label: "E-Sahakara Ads",
    content: (
      <div className="flex flex-col md:flex-row-reverse items-center gap-8 p-6">
        <div className="flex-1 space-y-3">
          <p>Promote your products to millions.</p>
          <ul className="list-disc ml-6">
            <li>Higher visibility</li>
            <li>Track performance</li>
            <li>Targeted promotions</li>
          </ul>
        </div>
        <div className="flex-1 flex justify-center">
          <img
            src={ImgAds}
            alt="Ads"
            className="w-2/3 max-w-xs rounded-lg shadow-md object-contain"
          />
        </div>
      </div>
    ),
  },
  {
    id: "festivals",
    label: "Shopping Festivals",
    content: (
      <div className="flex flex-col md:flex-row items-center gap-8 p-6">
        <div className="flex-1 space-y-3">
          <p>Participate in shopping festivals to boost your sales.</p>
          <ul className="list-disc ml-6">
            <li>Seasonal campaigns</li>
            <li>Exclusive discounts</li>
            <li>Reach new customers</li>
          </ul>
        </div>
        <div className="flex-1 flex justify-center">
          <img
            src={ImgFestivals}
            alt="Festivals"
            className="w-2/3 max-w-xs rounded-lg shadow-md object-contain"
          />
        </div>
      </div>
    ),
  },
];

const Grow = () => {
  const location = useLocation();
  const sectionRefs = useRef({});
  const [activeSection, setActiveSection] = useState("insights");

  const scrollToSection = (id) => {
    const el = sectionRefs.current[id];
    if (el) el.scrollIntoView({ behavior: "smooth", block: "start" });
    setActiveSection(id);
  };

  // Scroll to hash if exists
  useEffect(() => {
    const hash = location.hash.replace("#", "");
    if (hash) scrollToSection(hash);
  }, [location]);

  // Highlight active section while scrolling
  useEffect(() => {
    const onScroll = () => {
      const positions = Object.keys(sectionRefs.current).map((id) => ({
        id,
        top: Math.abs(sectionRefs.current[id].getBoundingClientRect().top),
      }));
      positions.sort((a, b) => a.top - b.top);
      if (positions[0] && positions[0].id !== activeSection)
        setActiveSection(positions[0].id);
    };
    window.addEventListener("scroll", onScroll);
    return () => window.removeEventListener("scroll", onScroll);
  }, [activeSection]);

  return (
    <div className="flex flex-col min-h-screen bg-gray-50 font-sans">
      {/* Navbar */}
      <UniversalNavbar
        growItems={sections.map((s) => ({
          label: s.label,
          path: "/grow",
          id: s.id,
        }))}
      />

      {/* Seller Slide */}
      <SellerSlide />

      <div className="flex flex-col md:flex-row px-5 py-8 gap-8">
        {/* Sidebar */}
        <aside className="hidden md:flex flex-col w-64 space-y-2 sticky top-20 h-[calc(100vh-5rem)]">
          {sections.map((section) => (
            <button
              key={section.id}
              onClick={() => scrollToSection(section.id)}
              className={`w-full text-left px-3 py-2 rounded-lg transition-all duration-300 ${
                activeSection === section.id
                  ? "bg-orange-100 text-orange-600 font-semibold shadow-md scale-105"
                  : "hover:bg-gray-100 text-gray-700"
              }`}
            >
              {section.label}
            </button>
          ))}
        </aside>

        {/* Content */}
        <main className="flex-1 space-y-24">
          {sections.map((section) => (
            <motion.section
              key={section.id}
              ref={(el) => (sectionRefs.current[section.id] = el)}
              className="scroll-mt-28"
              initial={{ opacity: 0, y: 40 }}
              whileInView={{ opacity: 1, y: 0 }}
              viewport={{ once: false, amount: 0.2 }}
              transition={{ duration: 0.6 }}
            >
              <h2 className="text-2xl font-bold mb-4 text-gray-900">
  {section.label}
</h2>

              <div className="text-gray-800 text-lg leading-relaxed">
                {section.content}
              </div>
            </motion.section>
          ))}
        </main>
      </div>
    </div>
  );
};

export default Grow;
