import { useState } from "react";
import { useNavigate } from "react-router-dom";
import toast from "react-hot-toast";

const LoginModal = ({ isOpen, onClose, onSuccess }) => {
  const navigate = useNavigate();
  const [step, setStep] = useState(1);
  const [loginData, setLoginData] = useState({ user: "", password: "", otp: "" });
  const [loading, setLoading] = useState(false);

  const handleUserNext = (e) => {
    e.preventDefault();
    if (!loginData.user.trim()) {
      toast.error("Please enter your email or phone number");
      return;
    }
    setLoading(true);
    setTimeout(() => {
      setStep(2);
      setLoading(false);
    }, 800);
  };

  const handlePasswordNext = (e) => {
    e.preventDefault();
    if (!loginData.password.trim()) {
      toast.error("Please enter your password");
      return;
    }
    setLoading(true);
    setTimeout(() => {
      setStep(3);
      setLoading(false);
    }, 800);
  };

  const handleOTPVerify = (e) => {
    e.preventDefault();
    if (!loginData.otp.trim()) {
      toast.error("Please enter OTP");
      return;
    }
    setLoading(true);
    setTimeout(() => {
      toast.success("Login successful!");
      setStep(1);
      setLoginData({ user: "", password: "", otp: "" });
      setLoading(false);
      onSuccess(); 
      onClose();
      navigate("/learn"); // keep nav visible after login
    }, 800);
  };

  if (!isOpen) return null;

  return (
    <div className="fixed inset-0 backdrop-blur-sm bg-black/30 flex items-center justify-center z-50">
      <div className="bg-white rounded-xl p-8 w-96 relative shadow-2xl transition-all duration-300">
        <button 
          onClick={onClose} 
          className="absolute top-3 right-3 text-gray-500 hover:text-gray-900 text-xl font-bold"
        >
          ×
        </button>

        {step === 1 && (
          <form onSubmit={handleUserNext} className="flex flex-col space-y-4">
            <h2 className="text-2xl font-bold text-gray-800">Login</h2>
            <input
              type="text"
              placeholder="Email or Phone"
              className="border border-gray-300 rounded-lg p-3 focus:outline-none focus:ring-2 focus:ring-orange-400"
              value={loginData.user}
              onChange={(e) => setLoginData({ ...loginData, user: e.target.value })}
            />
            <button 
              type="submit" 
              className="bg-orange-500 hover:bg-orange-600 text-white font-semibold py-3 rounded-lg transition-all"
            >
              {loading ? "Checking..." : "Next"}
            </button>

            {/* Register Button */}
            <button
              type="button"
              onClick={() => navigate("/startselling")}
              className="bg-white border border-orange-500 text-orange-500 font- py-3 rounded-lg hover:bg-orange-50 transition-all"
            >
              Register for a new account
            </button>

            {/* Disclaimer */}
            <p className="text-xs text-gray-500 text-center mt-2">
              By logging in or registering, you agree to the{" "}
              <span 
                className="text-orange-500 cursor-pointer hover:underline"
                onClick={() => navigate("/terms-privacy")}
              >
                E-Sahakara users terms and privacy policies
              </span>.
            </p>
          </form>
        )}

        {step === 2 && (
          <form onSubmit={handlePasswordNext} className="flex flex-col space-y-4">
            <h2 className="text-xl font-semibold text-gray-800">Enter Password</h2>
            <input
              type="password"
              placeholder="Password"
              className="border border-gray-300 rounded-lg p-3 focus:outline-none focus:ring-2 focus:ring-orange-400"
              value={loginData.password}
              onChange={(e) => setLoginData({ ...loginData, password: e.target.value })}
            />
            <button 
              type="submit" 
              className="bg-orange-500 hover:bg-orange-600 text-white font-semibold py-3 rounded-lg transition-all"
            >
              {loading ? "Verifying..." : "Next"}
            </button>
          </form>
        )}

        {step === 3 && (
          <form onSubmit={handleOTPVerify} className="flex flex-col space-y-4">
            <h2 className="text-xl font-semibold text-gray-800">Verify OTP</h2>
            <input
              type="text"
              placeholder="OTP"
              className="border border-gray-300 rounded-lg p-3 focus:outline-none focus:ring-2 focus:ring-orange-400"
              value={loginData.otp}
              onChange={(e) => setLoginData({ ...loginData, otp: e.target.value })}
            />
            <button 
              type="submit" 
              className="bg-orange-500 hover:bg-orange-600 text-white font-semibold py-3 rounded-lg transition-all"
            >
              {loading ? "Verifying..." : "Login"}
            </button>
          </form>
        )}
      </div>
    </div>
  );
};

export default LoginModal;
