import React, { useState } from "react";
import { Slider } from "@/components/ui/slider";
import { Checkbox } from "@/components/ui/checkbox";
import { Input } from "@/components/ui/input";
import { Star } from "lucide-react";
import { Button } from "@/components/ui/button";

const FiltersSidebar = ({ brands, filters, setFilters }) => {
    const [brandSearch, setBrandSearch] = useState("");
    const [collapsed, setCollapsed] = useState(false);

    const toggleBrand = (brand) => {
        setFilters((prev) => ({
            ...prev,
            brand: prev.brand.includes(brand)
                ? prev.brand.filter((b) => b !== brand)
                : [...prev.brand, brand],
        }));
    };

    const filteredBrands = brands.filter((b) =>
        b.toLowerCase().includes(brandSearch.toLowerCase())
    );

    return (
        <div className="p-5 space-y-8 bg-white min-h-screen shadow-sm sticky top-16">

            {/* PRICE */}
            <div>
                <h3 className="font-semibold mb-3 text-lg">Price</h3>
                <Slider
                    defaultValue={filters.price}
                    max={510000}
                    step={500}
                    onValueChange={(v) =>
                        setFilters((prev) => ({ ...prev, price: v }))
                    }
                />
                <p className="text-sm text-gray-600 mt-2">
                    ₹{filters.price[0]} – ₹{filters.price[1]}
                </p>
            </div>

            {/* BRAND */}
            <div>
                <div className="flex items-center justify-between">
                    <h3 className="font-semibold mb-3 text-lg">Brand</h3>
                    <button
                        className="text-sm text-blue-600"
                        onClick={() => setCollapsed(!collapsed)}
                    >
                        {collapsed ? "Show" : "Hide"}
                    </button>
                </div>

                {!collapsed && (
                    <>
                        <Input
                            placeholder="Search brand..."
                            value={brandSearch}
                            onChange={(e) => setBrandSearch(e.target.value)}
                            className="mb-3"
                        />

                        <div className="flex gap-2 mb-3">
                            <Button
                                variant="outline"
                                size="sm"
                                onClick={() =>
                                    setFilters((prev) => ({
                                        ...prev,
                                        brand: [...brands],
                                    }))
                                }
                            >
                                Select All
                            </Button>
                            <Button
                                variant="outline"
                                size="sm"
                                onClick={() =>
                                    setFilters((prev) => ({ ...prev, brand: [] }))
                                }
                            >
                                Clear
                            </Button>
                        </div>

                        <div className="max-h-48 overflow-y-auto pr-2 space-y-2">
                            {filteredBrands.map((brand) => (
                                <div
                                    key={brand}
                                    className="flex items-center gap-2"
                                >
                                    <Checkbox
                                        checked={filters.brand.includes(brand)}
                                        onCheckedChange={() =>
                                            toggleBrand(brand)
                                        }
                                    />

                                    <span className="text-sm">{brand}</span>
                                </div>
                            ))}
                        </div>
                    </>
                )}
            </div>

            {/* RATING */}
            <div>
                <h3 className="font-semibold mb-3 text-lg">Customer Ratings</h3>

                {[4, 3, 2].map((value) => (
                    <div
                        key={value}
                        className="flex items-center gap-2 mb-2 cursor-pointer"
                        onClick={() =>
                            setFilters((prev) => ({
                                ...prev,
                                rating:
                                    prev.rating === value ? null : value,
                            }))
                        }
                    >
                        <Checkbox checked={filters.rating === value} />

                        <div className="flex items-center gap-1 text-sm">
                            {[...Array(value)].map((_, i) => (
                                <Star
                                    key={i}
                                    size={14}
                                    fill="gold"
                                    className="text-yellow-400"
                                />
                            ))}
                            <span>& Up</span>
                        </div>
                    </div>
                ))}
            </div>

            {/* DISCOUNT */}
            <div>
                <h3 className="font-semibold mb-3 text-lg">Discount</h3>

                {[10, 20, 30, 50].map((d) => (
                    <div
                        key={d}
                        className="flex items-center gap-2 mb-1 cursor-pointer"
                        onClick={() =>
                            setFilters((prev) => ({
                                ...prev,
                                discount:
                                    prev.discount === d ? null : d,
                            }))
                        }
                    >
                        <Checkbox checked={filters.discount === d} />
                        <span className="text-sm">{d}% or More</span>
                    </div>
                ))}
            </div>

            {/* DELIVERY */}
            <div>
                <h3 className="font-semibold mb-3 text-lg">Delivery</h3>

                <div className="flex items-center gap-2">
                    <Checkbox
                        checked={filters.freeDelivery}
                        onCheckedChange={(v) =>
                            setFilters((prev) => ({
                                ...prev,
                                freeDelivery: v,
                            }))
                        }
                    />
                    <span className="text-sm">Free Delivery</span>
                </div>
            </div>

            {/* IN STOCK */}
            <div>
                <h3 className="font-semibold mb-3 text-lg">Availability</h3>

                <div className="flex items-center gap-2">
                    <Checkbox
                        checked={filters.inStock}
                        onCheckedChange={(v) =>
                            setFilters((prev) => ({
                                ...prev,
                                inStock: v,
                            }))
                        }
                    />
                    <span className="text-sm">In Stock Only</span>
                </div>
            </div>

            {/* NEW ARRIVALS */}
            <div>
                <h3 className="font-semibold mb-3 text-lg">New Arrivals</h3>

                <div className="flex flex-col gap-2">
                    {[30, 90].map((days) => (
                        <div
                            key={days}
                            className="flex items-center gap-2 cursor-pointer"
                            onClick={() =>
                                setFilters((prev) => ({
                                    ...prev,
                                    newArrival:
                                        prev.newArrival === days
                                            ? null
                                            : days,
                                }))
                            }
                        >
                            <Checkbox
                                checked={filters.newArrival === days}
                            />
                            <span className="text-sm">
                                Last {days} Days
                            </span>
                        </div>
                    ))}
                </div>
            </div>

            {/* VERIFIED SELLER */}
            <div>
                <h3 className="font-semibold mb-3 text-lg">Seller</h3>

                <div className="flex items-center gap-2">
                    <Checkbox
                        checked={filters.verifiedSeller}
                        onCheckedChange={(v) =>
                            setFilters((prev) => ({
                                ...prev,
                                verifiedSeller: v,
                            }))
                        }
                    />
                    <span className="text-sm">Verified Seller Only</span>
                </div>
            </div>
        </div>
    );
};

export default FiltersSidebar;
