import jwt from 'jsonwebtoken';
import bcrypt from 'bcrypt';
import validator from 'validator';
import userModel from '../models/userModel.js';

// Create JWT token
const createToken = (id) => {
    return jwt.sign({ id }, process.env.JWT_SECRET, { expiresIn: '1h' });
};

// ====================== USER SIGNUP ======================
const userSignUp = async (req, res) => {
    try {
        const { userName, email, password } = req.body;

        // Check if user exists
        const existingUser = await userModel.findOne({ email });
        if (existingUser) {
            return res.status(400).json({ success: false, message: 'User already exists' });
        }

        // Validate email
        if (!validator.isEmail(email)) {
            return res.status(400).json({ success: false, message: 'Please enter a valid email' });
        }

        // Validate password strength
        if (!validator.isStrongPassword(password, { minLength: 8 })) {
            return res.status(400).json({
                success: false,
                message: 'Password is too weak. Use at least 8 characters with uppercase, lowercase, number, and symbol',
            });
        }

        // Hash password
        const hashedPassword = await bcrypt.hash(password, 10);

        // Create new user
        const newUser = new userModel({
            userName,
            email,
            password: hashedPassword,
        });

        const savedUser = await newUser.save();

        // Generate token
        const userToken = createToken(savedUser._id);

        // Set token in HTTP-only cookie
        res.cookie('userToken', userToken, {
            httpOnly: true,
            secure: process.env.NODE_ENV === 'production',
            sameSite: 'Strict',
            maxAge: 7 * 24 * 60 * 60 * 1000, // 7 days
        });

        return res.status(201).json({ success: true, message: 'User registered successfully' });
    } catch (error) {
        console.error('Signup Error:', error);
        return res.status(500).json({ success: false, message: 'Server error' });
    }
};

// ====================== USER LOGIN ======================
const userLogin = async (req, res) => {
    try {
        const { email, password } = req.body;

        // Validate input
        if (!email || !password) {
            return res.status(400).json({ success: false, message: 'Email and password are required' });
        }

        // Find user
        const user = await userModel.findOne({ email });
        if (!user || !user.password) {
            return res.status(401).json({ success: false, message: 'No account found with this email. Please sign up first or check your email/password' });
        }

        // Compare password
        const isMatch = await bcrypt.compare(password, user.password);
        if (!isMatch) {
            return res.status(401).json({ success: false, message: 'Invalid Credentials' });
        }

        // Generate token
        const userToken = createToken(user._id);

        // Set token in HTTP-only cookie
        res.cookie('userToken', userToken, {
            httpOnly: true,
            secure: process.env.NODE_ENV === 'production',
            sameSite: 'Strict',
            maxAge: 7 * 24 * 60 * 60 * 1000, // 7 days
        });

        return res.status(200).json({ success: true, userToken, message: 'Login successful' });
    } catch (error) {
        console.error('Login Error:', error);
        return res.status(500).json({ success: false, message: 'Server error' });
    }
};

// ====================== USER LOGOUT ======================
const userLogout = async (req, res) => {
    try {
        // Clear cookie
        res.clearCookie('userToken', {
            httpOnly: true,
            secure: process.env.NODE_ENV === 'production',
            sameSite: 'Strict',
        });

        return res.status(200).json({ success: true, message: 'User logged out successfully' });
    } catch (error) {
        console.error('Logout Error:', error);
        return res.status(500).json({ success: false, message: 'Logout failed' });
    }
};

export { userSignUp, userLogin, userLogout };
