import { useState } from "react";
import { LineChart, Line, XAxis, YAxis, CartesianGrid, Tooltip, ResponsiveContainer } from "recharts";
import { motion } from "framer-motion";

export default function AdminDashboard() {
    const [stats] = useState({ users: 120, orders: 85, products: 45, vendors: 12, sales: "$12,500" });
    const [salesData] = useState([
        { month: "Jan", sales: 4000 }, { month: "Feb", sales: 3200 },
        { month: "Mar", sales: 5000 }, { month: "Apr", sales: 4300 },
        { month: "May", sales: 6100 }, { month: "Jun", sales: 7500 },
        { month: "Jul", sales: 6900 }, { month: "Aug", sales: 5000 },
        { month: "Sep", sales: 5500 }, { month: "Oct", sales: 7200 },
        { month: "Nov", sales: 8100 }, { month: "Dec", sales: 9000 },
    ]);
    const [recentOrders] = useState([
        { id: 1001, customer: "John Doe", amount: 250.75, status: "Pending" },
        { id: 1002, customer: "Jane Smith", amount: 499.99, status: "Shipped" },
        { id: 1003, customer: "Mark Wilson", amount: 89.0, status: "Delivered" },
        { id: 1004, customer: "Alice Johnson", amount: 320.0, status: "Pending" },
        { id: 1005, customer: "Robert Brown", amount: 150.5, status: "Shipped" },
        { id: 1006, customer: "Emily Davis", amount: 420.25, status: "Delivered" },
    ]);

    const cardVariant = {
        hidden: { opacity: 0, y: 20 },
        visible: { opacity: 1, y: 0, transition: { duration: 0.5 } }
    };
    const rowVariant = {
        hidden: { opacity: 0, x: -20 },
        visible: i => ({ opacity: 1, x: 0, transition: { delay: i * 0.1 } })
    };

    return (
        <div className="min-h-screen bg-gray-50 p-4 md:p-8">
            <motion.h1
                className="text-2xl md:text-3xl text-orange-600 font-semibold mb-6"
                initial={{ opacity: 0 }}
                animate={{ opacity: 1 }}
                transition={{ duration: 0.6 }}
            >
                Dashboard Overview
            </motion.h1>

            {/* Top Stats */}
            <div className="grid grid-cols-1 sm:grid-cols-2 lg:grid-cols-4 gap-4 md:gap-6 mb-8">
                {[
                    { title: "Users", value: stats.users, color: "bg-blue-600" },
                    { title: "Orders", value: stats.orders, color: "bg-green-600" },
                    { title: "Products", value: stats.products, color: "bg-purple-600" },
                    { title: "Vendors", value: stats.vendors, color: "bg-yellow-500" },
                ].map((stat, idx) => (
                    <motion.div
                        key={stat.title}
                        className={`${stat.color} text-white rounded-xl p-6 shadow-md cursor-pointer`}
                        variants={cardVariant}
                        initial="hidden"
                        animate="visible"
                        transition={{ delay: idx * 0.2 }}
                        whileHover={{ scale: 1.05, boxShadow: "0px 10px 20px rgba(0,0,0,0.15)" }}
                    >
                        <h2 className="text-sm md:text-base font-medium mb-1">{stat.title}</h2>
                        <p className="text-3xl md:text-4xl font-bold">{stat.value}</p>
                    </motion.div>
                ))}
            </div>

            {/* Sales Chart */}
            <motion.div
                className="bg-white p-6 rounded-xl shadow-md mb-8"
                initial={{ opacity: 0 }}
                animate={{ opacity: 1 }}
                transition={{ duration: 0.8 }}
                whileHover={{ scale: 1.01, boxShadow: "0px 15px 30px rgba(0,0,0,0.1)" }}
            >
                <div className="flex flex-col sm:flex-row justify-between items-start sm:items-center mb-6">
                    <h2 className="text-xl md:text-2xl font-semibold mb-2 sm:mb-0">Sales Overview</h2>
                    <span className="text-base md:text-lg text-gray-600 font-medium">Total: {stats.sales}</span>
                </div>
                <div className="w-full h-80">
                    <ResponsiveContainer width="100%" height="100%">
                        <LineChart data={salesData} margin={{ top: 10, right: 30, left: 0, bottom: 0 }}>
                            <CartesianGrid strokeDasharray="3 3" stroke="#e5e7eb" />
                            <XAxis 
                                dataKey="month" 
                                tick={{ fontSize: 12 }}
                                stroke="#6b7280"
                            />
                            <YAxis 
                                tick={{ fontSize: 12 }}
                                stroke="#6b7280"
                                tickFormatter={(value) => `$${value}`}
                            />
                            <Tooltip 
                                formatter={(value) => [`$${value}`, 'Sales']}
                                contentStyle={{ 
                                    backgroundColor: '#fff', 
                                    border: '1px solid #e5e7eb',
                                    borderRadius: '8px',
                                    padding: '8px 12px'
                                }}
                            />
                            <Line
                                type="monotone"
                                dataKey="sales"
                                stroke="#2563eb"
                                strokeWidth={3}
                                dot={{ fill: '#2563eb', r: 4 }}
                                activeDot={{ r: 6 }}
                                animationDuration={1500}
                            />
                        </LineChart>
                    </ResponsiveContainer>
                </div>
            </motion.div>

            {/* Recent Orders */}
            <motion.div
                className="bg-white p-6 rounded-xl shadow-md overflow-hidden"
                initial={{ opacity: 0 }}
                animate={{ opacity: 1 }}
                transition={{ duration: 0.8 }}
            >
                <h2 className="text-xl md:text-2xl font-semibold mb-4">Recent Orders</h2>
                <div className="overflow-x-auto">
                    <table className="w-full">
                        <thead className="bg-gray-100">
                            <tr>
                                <th className="py-3 px-4 text-left text-sm font-semibold text-gray-700 w-20">ID</th>
                                <th className="py-3 px-4 text-left text-sm font-semibold text-gray-700">Customer</th>
                                <th className="py-3 px-4 text-left text-sm font-semibold text-gray-700 w-32">Amount</th>
                                <th className="py-3 px-4 text-left text-sm font-semibold text-gray-700 w-32">Status</th>
                            </tr>
                        </thead>
                        <tbody>
                            {recentOrders.map((o, i) => (
                                <motion.tr
                                    key={o.id}
                                    className="border-b border-gray-100 hover:bg-gray-50 cursor-pointer"
                                    custom={i}
                                    variants={rowVariant}
                                    initial="hidden"
                                    animate="visible"
                                    whileHover={{ backgroundColor: "#f9fafb" }}
                                >
                                    <td className="py-3 px-4 text-sm text-gray-800">{o.id}</td>
                                    <td className="py-3 px-4 text-sm text-gray-800">{o.customer}</td>
                                    <td className="py-3 px-4 text-sm text-gray-800 font-medium">${o.amount.toFixed(2)}</td>
                                    <td className="py-3 px-4">
                                        <span className={`inline-block px-3 py-1 rounded-full text-xs font-semibold ${
                                            o.status === "Delivered" 
                                                ? "bg-green-100 text-green-700" 
                                                : o.status === "Shipped" 
                                                ? "bg-blue-100 text-blue-700" 
                                                : "bg-yellow-100 text-yellow-700"
                                        }`}>
                                            {o.status}
                                        </span>
                                    </td>
                                </motion.tr>
                            ))}
                        </tbody>
                    </table>
                </div>
            </motion.div>
        </div>
    );
}