import { useState } from 'react';
import Breadcrumbs from "../../../../commonComponents/breadcrumbs/Breadcrumbs";
import { UploadCloud } from 'lucide-react';

export default function ImportProducts() {
    const [step, setStep] = useState(1);
    const [file, setFile] = useState(null);
    const [error, setError] = useState("");
    const [dragActive, setDragActive] = useState(false);


    const breadcrumbPaths = [
        { label: 'Products', to: '/admin/products' },
        { label: 'Import' },
    ];

    const handleNext = () => {
        // Step 2 validation before going to Step 3
        if (step === 2 && !file) {
            setError(" Please upload a files before continuing.");
            return;
        }

        setError(""); // Clear any previous errors
        if (step < 3) setStep(step + 1);
    };

    const handlePrev = () => {
        setError("");
        if (step > 1) setStep(step - 1);
    };

    const handleFileChange = (e) => {
        setFile(e.target.files[0]);
        setError(""); // clear error once user uploads
    };

    return (<>
        <Breadcrumbs paths={breadcrumbPaths} />
        <div className="min-h-screen bg-gray-100 p-4 sm:p-6">

            <div className="bg-white p-6 rounded-lg shadow max-w-4xl mx-auto">
                <h1 className="text-2xl font-bold mb-4">Prepare CSV file for import</h1>

                {/* Step Indicators */}
                <div className="relative p-5 flex justify-between items-center mb-8 max-w-md mx-auto">
                    {[1, 2, 3].map((s, i) => (
                        <div key={s} className="flex-1 flex items-center">
                            <div className="relative flex flex-col items-center w-full">
                                <div
                                    className={`w-10 h-10 flex items-center justify-center rounded-full font-semibold text-sm shadow-sm transition-all duration-300 
                                    ${step === s
                                            ? 'bg-orange-500 text-white scale-110 ring-4 ring-orange-100'
                                            : step > s
                                                ? 'bg-green-500 text-white'
                                                : 'bg-gray-200 text-gray-500'
                                        }`}
                                >
                                    {step > s ? '✓' : s}
                                </div>
                                <span
                                    className={`mt-2 text-xs font-medium transition-colors duration-300 
                                    ${step === s
                                            ? 'text-orange-500'
                                            : step > s
                                                ? 'text-green-600'
                                                : 'text-gray-400'
                                        }`}
                                >
                                    Step {s}
                                </span>
                            </div>

                            {i < 2 && (
                                <div
                                    className={`flex-1 h-1 mx-2 transition-all duration-500 rounded-full 
                                    ${step > s ? 'bg-green-500' : 'bg-gray-300'}`}
                                ></div>
                            )}
                        </div>
                    ))}
                </div>

                {/* Step Content */}
                {step === 1 && (
                    <div>
                        <p className='p-5 text-gray-500'>The import tool lets you create or update physical products in your store. To get started, prepare a CSV file compatible with our template</p>
                        <ul className="list-disc list-inside p-4 text-gray-500 mb-4">
                            <li>
                                To import new products, download the CSV template.{" "}
                                <a
                                    href="/templates/products_template.csv"
                                    download
                                    className="text-blue-600 font-medium hover:underline ml-1"
                                >
                                    Download Template
                                </a>
                            </li>
                            <li>To update existing products, export your products to CSV.</li>
                            <li>Fill in the CSV file with your product details. Each file can have up to 10,000 rows and weigh up to 40 MB.</li>
                            <li>Come back here and continue to upload your file.</li>
                        </ul>
                        <button
                            onClick={handleNext}
                            className="px-4 py-2 bg-orange-500 text-white rounded hover:bg-orange-600"
                        >
                            Next
                        </button>
                    </div>
                )}

                {step === 2 && (
                    <div>
                        <h2 className="text-lg font-semibold mb-4">Upload CSV File</h2>

                        {/* Drag & Drop Upload Box */}
                        <div
                            onClick={() => document.getElementById("fileInput").click()}
                            onDragOver={(e) => {
                                e.preventDefault();
                                e.stopPropagation();
                                setDragActive(true);
                            }}
                            onDragLeave={(e) => {
                                e.preventDefault();
                                e.stopPropagation();
                                setDragActive(false);
                            }}
                            onDrop={(e) => {
                                e.preventDefault();
                                e.stopPropagation();
                                setDragActive(false);
                                if (e.dataTransfer.files && e.dataTransfer.files[0]) {
                                    const uploadedFile = e.dataTransfer.files[0];
                                    if (uploadedFile.type === "text/csv") {
                                        setFile(uploadedFile);
                                        setError("");
                                    } else {
                                        setError(" Only CSV files are allowed.");
                                    }
                                }
                            }}
                            className={`border-2 border-dashed rounded-xl p-8 text-center cursor-pointer hover:bg-orange-100 transition-all ${dragActive ? "border-orange-400 bg-orange-50" : "border-gray-300 bg-gray-50"
                                }`}
                        >
                            <UploadCloud className="w-10 h-10 mx-auto text-orange-500 mb-3" />
                            <p className="text-gray-500 font-medium mb-1">
                                Drag & drop your CSV file here, or{" "}
                                <span className="text-orange-500 underline">browse</span>
                            </p>
                            <p className="text-sm text-gray-500">Only .csv files are accepted</p>

                            <input
                                id="fileInput"
                                type="file"
                                accept=".csv"
                                onChange={handleFileChange}
                                className="hidden"
                            />
                        </div>

                        {/* Show selected file */}
                        {file && (
                            <div className="mt-4 bg-green-50 border border-green-200 rounded-lg p-3 flex items-center justify-between">
                                <p className="text-sm text-green-800 truncate">
                                    <span className="font-medium">Selected file:</span> {file.name}
                                </p>
                                <button
                                    onClick={() => setFile(null)}
                                    className="text-xs text-red-600 hover:underline ml-2"
                                >
                                    Remove
                                </button>
                            </div>
                        )}

                        {/* Error message */}
                        {error && (
                            <p className="text-red-500 text-sm font-medium mt-3 animate-pulse">
                                {error}
                            </p>
                        )}

                        {/* Navigation Buttons */}
                        <div className="flex gap-2 mt-6">
                            <button
                                onClick={handlePrev}
                                className="px-4 py-2 bg-gray-200 rounded hover:bg-gray-300"
                            >
                                Back
                            </button>
                            <button
                                onClick={() => {
                                    if (!file) {
                                        setError(" Please upload a file before continuing.");
                                        return;
                                    }
                                    handleNext();
                                }}
                                className={`px-4 py-2 rounded text-white ${file
                                    ? "bg-orange-600 hover:bg-orange-700"
                                    : "bg-gray-300 cursor-not-allowed"
                                    }`}
                            >
                                Next
                            </button>
                        </div>
                    </div>
                )}


                {step === 3 && (
                    <div>
                        <h2 className="text-lg font-semibold mb-2">Confirm Import</h2>
                        <p className="text-gray-500 mb-4">
                            You are about to import the selected CSV file. Please confirm to proceed.
                        </p>
                        <div className="flex gap-2">
                            <button
                                onClick={handlePrev}
                                className="px-4 py-2 bg-gray-200 rounded hover:bg-gray-300"
                            >
                                Back
                            </button>
                            <button
                                onClick={() => alert('Import Started!')}
                                className="px-4 py-2 bg-orange-500 text-white rounded hover:bg-orange-600 flex items-center gap-2"
                            >
                                <UploadCloud className="w-5 h-5" />
                                Import
                            </button>
                        </div>
                    </div>
                )}
            </div>
        </div>
    </>
    );
}
