import { useState, useRef, useEffect } from 'react';
import {
  Search,
  Filter,
  Upload,
  SlidersHorizontal,
  ChevronDown,
  MoreHorizontal,
  Plus,
  DownloadCloud,
  UploadCloud
} from 'lucide-react';
import { useNavigate } from 'react-router-dom';


export default function ProductManagement() {
  const navigate = useNavigate();
  const [allProducts] = useState([
    {
      id: 1,
      name: 'Ceramic Flower Vase',
      variants: null,
      type: 'Physical',
      sku: '364215376135191',
      price: 270.0,
      inventory: 'In stock',
      image: '🏺',
    },
    {
      id: 2,
      name: 'Minimalist Tote Bag',
      variants: '2 variants',
      type: 'Physical',
      sku: '',
      price: 20.0,
      inventory: 'In stock',
      image: '👜',
    },
    {
      id: 3,
      name: 'Hydrating Eye Serum ...',
      variants: null,
      type: 'Physical',
      sku: '364115376135191',
      price: 56.0,
      inventory: 'Out of',
      image: '🧴',
    },
    {
      id: 4,
      name: 'Knitted Golf Sweater',
      variants: '12 variants',
      type: 'Physical',
      sku: '',
      price: 275.0,
      inventory: 'In stock',
      image: '👕',
    },
    {
      id: 5,
      name: 'Wireless Earbuds',
      variants: '3 variants',
      type: 'Physical',
      sku: '364215376135192',
      price: 150.0,
      inventory: 'In stock',
      image: '🎧',
    },
  ]);

  const products = allProducts;

  const [showMoreActions, setShowMoreActions] = useState(false);
  const [open, setOpen] = useState(false)
  const moreActionsRef = useRef(null);
  const dropDownRef = useRef(null)

  useEffect(() => {
    function handleClickOutside(event) {
      if (moreActionsRef.current && !moreActionsRef.current.contains(event.target)) {
        setShowMoreActions(false);
      }
    }
    document.addEventListener('mousedown', handleClickOutside);
    return () => document.removeEventListener('mousedown', handleClickOutside);
  }, []);

  return (
    <div className="min-h-screen bg-gray-100 p-3 sm:p-4 md:p-6">
      <div className="max-w-7xl mx-auto">
        {/* Header */}
        <div className="mb-4 md:mb-6">
          <div className="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-3 mb-2 relative">
            <div>
              <h1 className="text-2xl sm:text-3xl md:text-4xl font-semibold">
                Products <span className="text-orange-500">{products.length}</span>
              </h1>
            </div>

            {/* Buttons */}
            <div className="flex gap-2 sm:gap-3 relative" ref={moreActionsRef}>
              {/* More Actions Button */}
              <button
                onClick={() => setShowMoreActions(!showMoreActions)}
                className="hidden sm:flex px-3 md:px-4 py-2 bg-white rounded-lg border border-gray-200 hover:bg-gray-50 items-center gap-2"
              >
                <span className="font-medium text-sm md:text-base">More Actions</span>
                <ChevronDown className="w-3 h-3 md:w-4 md:h-4" />
              </button>

              {/* Dropdown Card */}
              {showMoreActions && (
                <div className="absolute top-full right-0 mt-2 w-64 bg-white border border-gray-200 rounded-lg shadow-lg z-10">
                  <button onClick={() => navigate('/admin/products/import')} className="w-full flex items-start gap-3 px-4 py-3 hover:bg-gray-50 rounded-t-lg">
                    <UploadCloud className="w-6 h-6 text-blue-600 mt-1" />
                    <div className="text-left">
                      <p className="font-medium text-gray-900 text-sm">Import</p>
                      <p className="text-gray-500 text-xs">Import multiple products to your store</p>
                    </div>
                  </button>
                  <button className="w-full flex items-start gap-3 px-4 py-3 hover:bg-gray-50 rounded-b-lg">
                    <DownloadCloud className="w-6 h-6 text-green-600 mt-1" />
                    <div className="text-left">
                      <p className="font-medium text-gray-900 text-sm">Export</p>
                      <p className="text-gray-500 text-xs">Export your physical products to a CSV file</p>
                    </div>
                  </button>
                </div>
              )}

              {/* Mobile More Button */}
              <button className="sm:hidden p-2 bg-white rounded-lg border border-gray-200 hover:bg-gray-50">
                <MoreHorizontal className="w-4 h-4" />
              </button>

              {/* New Product Button */}
              <button onClick={() => setOpen(!open)} className="px-3 md:px-4 py-2 bg-orange-500 text-white rounded-lg hover:bg-orange-600 flex items-center gap-2">
                <Plus className="w-4 h-4 md:w-5 md:h-5" />
                <span className="font-medium text-sm md:text-base hidden sm:inline">New Product</span>
                <span className="font-medium text-sm sm:hidden">New</span>
              </button>

              {/* DropDown for New Products */}
              {open && (
                <div className="absolute top-full right-0 mt-2 w-64 bg-white border border-gray-200 rounded-lg shadow-lg z-10">
                  <button onClick={() => navigate('/admin/products/import')} className="w-full flex items-start gap-3 px-4 py-3 hover:bg-gray-50 rounded-t-lg">
                    <div className="text-left">
                      <p className="font-medium text-gray-900 text-sm">Physical Products</p>
                      <p className="text-gray-500 text-xs">Import multiple products to your store</p>
                    </div>
                  </button>
                  <button className="w-full flex items-start gap-3 px-4 py-3 hover:bg-gray-50 rounded-b-lg">
                    <div className="text-left">
                      <p className="font-medium text-gray-900 text-sm">Export</p>
                      <p className="text-gray-500 text-xs">Export your physical products to a CSV file</p>
                    </div>
                  </button>
                </div>
              )}
            </div>
          </div>
        </div>

        {/* Filters Bar */}
        <div className="bg-white rounded-lg shadow-sm overflow-hidden">
          <div className="p-3 md:p-4 border-b border-gray-200">
            {/* Desktop Filters */}
            <div className="hidden lg:flex items-center justify-between">
              <div className="flex items-center gap-4">
                <button className="flex items-center gap-2 text-gray-700 hover:bg-gray-50 px-3 py-2 rounded-lg">
                  <span>All products</span>
                  <span className="text-gray-500">({products.length})</span>
                  <ChevronDown className="w-3 h-3" />
                </button>
                <button className="flex items-center gap-2 text-blue-600 border-2 border-blue-600 px-3 py-2 rounded-lg hover:bg-blue-50">
                  <span className="font-medium">Manage View</span>
                  <ChevronDown className="w-3 h-3" />
                </button>
              </div>

              <div className="flex items-center gap-3">
                <button className="flex items-center gap-2 px-3 py-2 hover:bg-gray-50 rounded-lg">
                  <Filter className="w-4 h-4 text-gray-600" />
                  <span className="text-blue-600 font-medium">Filter</span>
                </button>
                <button className="p-2 hover:bg-gray-50 rounded-lg">
                  <Upload className="w-4 h-4 text-gray-600" />
                </button>
                <button className="p-2 hover:bg-gray-50 rounded-lg">
                  <SlidersHorizontal className="w-4 h-4 text-gray-600" />
                </button>
                <div className="relative">
                  <Search className="w-4 h-4 text-gray-400 absolute left-3 top-1/2 -translate-y-1/2" />
                  <input
                    type="text"
                    placeholder="Search..."
                    className="pl-10 pr-4 py-2 border border-gray-200 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 w-64"
                  />
                </div>
              </div>
            </div>

            {/* Mobile/Tablet Filters */}
            <div className="lg:hidden space-y-3">
              <div className="flex items-center gap-2">
                <button className="flex-1 flex items-center justify-center gap-2 text-gray-700 bg-gray-50 px-3 py-2 rounded-lg text-sm">
                  <span>All products ({products.length})</span>
                  <ChevronDown className="w-3 h-3" />
                </button>
                <button className="flex-1 flex items-center justify-center gap-2 text-blue-600 border-2 border-blue-600 px-3 py-2 rounded-lg text-sm">
                  <span className="font-medium">Manage View</span>
                  <ChevronDown className="w-3 h-3" />
                </button>
              </div>

              <div className="flex items-center gap-2">
                <div className="relative flex-1">
                  <Search className="w-4 h-4 text-gray-400 absolute left-3 top-1/2 -translate-y-1/2" />
                  <input
                    type="text"
                    placeholder="Search..."
                    className="w-full pl-10 pr-4 py-2 border border-gray-200 rounded-lg focus:outline-none focus:ring-2 focus:ring-blue-500 text-sm"
                  />
                </div>
                <button className="p-2 border border-gray-200 rounded-lg hover:bg-gray-50">
                  <Filter className="w-4 h-4 text-gray-600" />
                </button>
                <button className="p-2 border border-gray-200 rounded-lg hover:bg-gray-50">
                  <SlidersHorizontal className="w-4 h-4 text-gray-600" />
                </button>
              </div>
            </div>
          </div>

          {/* Table - Desktop */}
          <div className="hidden md:block overflow-x-auto">
            <table className="w-full">
              <thead>
                <tr className="bg-blue-50 border-b border-gray-200">
                  <th className="text-left p-4 w-12">
                    <input type="checkbox" className="w-4 h-4 rounded border-gray-300" />
                  </th>
                  <th className="text-left p-4 font-medium text-gray-700">Name</th>
                  <th className="text-left p-4 font-medium text-gray-700">Type</th>
                  <th className="text-left p-4 font-medium text-gray-700 hidden lg:table-cell">SKU</th>
                  <th className="text-left p-4 font-medium text-gray-700">Price</th>
                  <th className="text-left p-4 font-medium text-gray-700">Inventory</th>
                  <th className="w-12"></th>
                </tr>
              </thead>
              <tbody>
                {products.map((product) => (
                  <tr key={product.id} className="border-b border-gray-200 hover:bg-gray-50">
                    <td className="p-4">
                      <input type="checkbox" className="w-4 h-4 rounded border-gray-300" />
                    </td>
                    <td className="p-4">
                      <div className="flex items-center gap-3">
                        <div className="w-10 h-10 lg:w-12 lg:h-12 bg-gray-100 rounded-lg flex items-center justify-center text-xl lg:text-2xl">
                          {product.image}
                        </div>
                        <div>
                          <div className="font-medium text-gray-900 text-sm lg:text-base">{product.name}</div>
                          {product.variants && (
                            <div className="text-xs lg:text-sm text-gray-500">{product.variants}</div>
                          )}
                        </div>
                      </div>
                    </td>
                    <td className="p-4 text-gray-700 text-sm lg:text-base">{product.type}</td>
                    <td className="p-4 text-gray-700 text-sm lg:text-base hidden lg:table-cell">{product.sku}</td>
                    <td className="p-4 text-gray-900 text-sm lg:text-base">₹{product.price.toFixed(2)}</td>
                    <td className="p-4 text-gray-700 text-sm lg:text-base">{product.inventory}</td>
                    <td className="p-4">
                      <button className="p-1 hover:bg-gray-100 rounded">
                        <MoreHorizontal className="w-4 h-4 text-blue-600" />
                      </button>
                    </td>
                  </tr>
                ))}
              </tbody>
            </table>
          </div>

          {/* Cards - Mobile */}
          <div className="md:hidden divide-y divide-gray-200">
            {products.map((product) => (
              <div key={product.id} className="p-4 hover:bg-gray-50">
                <div className="flex gap-3">
                  <input type="checkbox" className="w-4 h-4 rounded border-gray-300 mt-1" />
                  <div className="w-16 h-16 bg-gray-100 rounded-lg flex items-center justify-center text-3xl flex-shrink-0">
                    {product.image}
                  </div>
                  <div className="flex-1 min-w-0">
                    <div className="flex items-start justify-between gap-2">
                      <div className="flex-1 min-w-0">
                        <h3 className="font-medium text-gray-900 truncate">{product.name}</h3>
                        {product.variants && (
                          <p className="text-sm text-gray-500">{product.variants}</p>
                        )}
                      </div>
                      <button className="p-1 hover:bg-gray-100 rounded flex-shrink-0">
                        <MoreHorizontal className="w-4 h-4 text-blue-600" />
                      </button>
                    </div>
                    <div className="mt-2 grid grid-cols-2 gap-2 text-sm">
                      <div>
                        <span className="text-gray-500">Type:</span>
                        <span className="ml-1 text-gray-900">{product.type}</span>
                      </div>
                      <div>
                        <span className="text-gray-500">Price:</span>
                        <span className="ml-1 text-gray-900 font-medium">₹{product.price.toFixed(2)}</span>
                      </div>
                      {product.sku && (
                        <div className="col-span-2">
                          <span className="text-gray-500">SKU:</span>
                          <span className="ml-1 text-gray-900 text-xs">{product.sku}</span>
                        </div>
                      )}
                      <div className="col-span-2">
                        <span className="text-gray-500">Status:</span>
                        <span className="ml-1 text-gray-900">{product.inventory}</span>
                      </div>
                    </div>
                  </div>
                </div>
              </div>
            ))}
          </div>
        </div>
      </div>
    </div>
  );
}
