import { useState, useEffect } from "react";
import { Link, useLocation } from "react-router-dom";
import { FontAwesomeIcon } from "@fortawesome/react-fontawesome";
import {
    faTachometerAlt,
    faBoxOpen,
    faShoppingCart,
    faUsers,
    faStore,
    faChevronDown,
    faChevronRight,
} from "@fortawesome/free-solid-svg-icons";
import logo from "../../assets/images/logo_plain.png";

export default function AdminSidebar() {
    const location = useLocation();
    const [openMenu, setOpenMenu] = useState(null);

    // Auto-expand catalog menu if current route is a catalog page
    useEffect(() => {
        if (
            location.pathname.startsWith("/admin/products") ||
            location.pathname.startsWith("/admin/inventory") ||
            location.pathname.startsWith("/admin/categories") ||
            location.pathname.startsWith("/admin/back-in-stock") ||
            location.pathname.startsWith("/admin/dropshipping")
        ) {
            setOpenMenu("Catalog");
        }
    }, [location.pathname]);

    const menuItems = [
        { name: "Dashboard", path: "/admin/dashboard", icon: faTachometerAlt },
        {
            name: "Catalog",
            icon: faBoxOpen,
            subItems: [
                { name: "Products", path: "/admin/products" },
                { name: "Inventory", path: "/admin/inventory" },
                { name: "Categories", path: "/admin/categories" },
                { name: "Back In Stock", path: "/admin/back-in-stock" },
                { name: "Dropshipping", path: "/admin/dropshipping" },
            ],
        },
        { name: "Orders", path: "/admin/orders", icon: faShoppingCart },
        { name: "Users", path: "/admin/users", icon: faUsers },
        { name: "Vendors", path: "/admin/vendors", icon: faStore },
    ];

    const toggleMenu = (name) => {
        setOpenMenu(openMenu === name ? null : name);
    };

    return (
        <div className="w-64 bg-white shadow-lg min-h-screen flex flex-col">
            {/* Logo */}
            <Link to="/admin/dashboard" className="flex items-center gap-2 p-5">
                <img src={logo} alt="Logo" className="w-10 h-10 object-contain" />
                <div className="flex flex-col">
                    <span className="font-bold text-2xl bg-gradient-to-r from-orange-600 to-yellow-400 bg-clip-text text-transparent">
                        ESAHAKARA
                    </span>
                    <span className="text-sm font-semibold text-gray-500">Products</span>
                </div>
            </Link>

            {/* Navigation */}
            <nav className="flex-1 overflow-y-auto p-4">
                <ul className="space-y-2">
                    {menuItems.map((item) => {
                        const isActive =
                            location.pathname === item.path ||
                            item.subItems?.some((sub) => location.pathname === sub.path);

                        return (
                            <li key={item.name}>
                                {item.subItems ? (
                                    // Parent with submenu
                                    <div
                                        onClick={() => toggleMenu(item.name)}
                                        className={`flex items-center justify-between gap-3 px-4 py-3 rounded-lg cursor-pointer transition-all duration-200 ${isActive
                                                ? "bg-orange-500 text-white shadow-md"
                                                : "text-gray-700 hover:bg-orange-50 hover:text-orange-600"
                                            }`}
                                    >
                                        <div className="flex items-center gap-3">
                                            <FontAwesomeIcon icon={item.icon} className="w-5 h-5" />
                                            <span className="font-medium">{item.name}</span>
                                        </div>
                                        <FontAwesomeIcon
                                            icon={openMenu === item.name ? faChevronDown : faChevronRight}
                                            className="w-4 h-4"
                                        />
                                    </div>
                                ) : (
                                    // Parent without submenu → clickable Link
                                    <Link
                                        to={item.path}
                                        className={`flex items-center gap-3 px-4 py-3 rounded-lg transition-all duration-200 ${isActive
                                                ? "bg-orange-500 text-white shadow-md"
                                                : "text-gray-700 hover:bg-orange-50 hover:text-orange-600"
                                            }`}
                                    >
                                        <FontAwesomeIcon icon={item.icon} className="w-5 h-5" />
                                        <span className="font-medium">{item.name}</span>
                                    </Link>
                                )}

                                {/* Submenu */}
                                {item.subItems && openMenu === item.name && (
                                    <ul className="ml-8 mt-2 space-y-2 border-l-2 border-orange-100 pl-3">
                                        {item.subItems.map((sub) => {
                                            const isSubActive = location.pathname === sub.path;
                                            return (
                                                <li key={sub.path}>
                                                    <Link
                                                        to={sub.path}
                                                        className={`block px-3 py-2 rounded-md text-sm transition-all duration-150 ${isSubActive
                                                                ? "bg-orange-100 text-orange-600 font-semibold"
                                                                : "text-gray-600 hover:bg-orange-50 hover:text-orange-600"
                                                            }`}
                                                    >
                                                        {sub.name}
                                                    </Link>
                                                </li>
                                            );
                                        })}
                                    </ul>
                                )}
                            </li>
                        );
                    })}
                </ul>
            </nav>
        </div>
    );
}
