import { useState } from "react";

export default function AdminProducts() {
    const [products, setProducts] = useState([
        { id: 1, name: "Wireless Headphones", price: 59.99, category: "Electronics", stock: 25 },
        { id: 2, name: "Smart Watch", price: 129.99, category: "Wearables", stock: 10 },
    ]);

    const [formData, setFormData] = useState({
        name: "",
        price: "",
        category: "",
        stock: "",
    });

    const [editId, setEditId] = useState(null);

    const handleChange = (e) => {
        const { name, value } = e.target;
        setFormData((prev) => ({ ...prev, [name]: value }));
    };

    const handleSubmit = (e) => {
        e.preventDefault();

        if (!formData.name || !formData.price || !formData.category || !formData.stock) {
            alert("Please fill in all fields");
            return;
        }

        if (editId) {
            // Edit product
            setProducts((prev) =>
                prev.map((p) => (p.id === editId ? { ...p, ...formData, price: parseFloat(formData.price), stock: parseInt(formData.stock) } : p))
            );
            setEditId(null);
        } else {
            // Add new product
            const newProduct = {
                id: products.length + 1,
                ...formData,
                price: parseFloat(formData.price),
                stock: parseInt(formData.stock),
            };
            setProducts((prev) => [...prev, newProduct]);
        }

        setFormData({ name: "", price: "", category: "", stock: "" });
    };

    const handleEdit = (product) => {
        setFormData({
            name: product.name,
            price: product.price,
            category: product.category,
            stock: product.stock,
        });
        setEditId(product.id);
    };

    const handleDelete = (id) => {
        if (window.confirm("Are you sure you want to delete this product?")) {
            setProducts((prev) => prev.filter((p) => p.id !== id));
        }
    };

    return (
        <div>
            <h1 className="text-2xl font-semibold mb-6">Product Management</h1>

            {/* Add / Edit Product Form */}
            <form onSubmit={handleSubmit} className="bg-white p-6 rounded-lg shadow-md mb-8 space-y-4">
                <h2 className="text-lg font-semibold mb-2">{editId ? "Edit Product" : "Add New Product"}</h2>
                <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                    <input
                        type="text"
                        name="name"
                        placeholder="Product Name"
                        value={formData.name}
                        onChange={handleChange}
                        className="border rounded p-2 w-full"
                    />
                    <input
                        type="number"
                        name="price"
                        placeholder="Price"
                        value={formData.price}
                        onChange={handleChange}
                        className="border rounded p-2 w-full"
                    />
                    <input
                        type="text"
                        name="category"
                        placeholder="Category"
                        value={formData.category}
                        onChange={handleChange}
                        className="border rounded p-2 w-full"
                    />
                    <input
                        type="number"
                        name="stock"
                        placeholder="Stock Quantity"
                        value={formData.stock}
                        onChange={handleChange}
                        className="border rounded p-2 w-full"
                    />
                </div>

                <div className="flex space-x-2">
                    <button type="submit" className="bg-blue-600 text-white px-6 py-2 rounded hover:bg-blue-700">
                        {editId ? "Update Product" : "Add Product"}
                    </button>
                    {editId && (
                        <button
                            type="button"
                            onClick={() => {
                                setEditId(null);
                                setFormData({ name: "", price: "", category: "", stock: "" });
                            }}
                            className="bg-gray-400 text-white px-6 py-2 rounded hover:bg-gray-500"
                        >
                            Cancel
                        </button>
                    )}
                </div>
            </form>

            {/* Products Table */}
            <div className="bg-white p-6 rounded-lg shadow-md">
                <h2 className="text-lg font-semibold mb-4">Product List</h2>
                <table className="min-w-full">
                    <thead className="bg-gray-200 text-gray-600 uppercase text-sm">
                        <tr>
                            <th className="py-3 px-6 text-left">Name</th>
                            <th className="py-3 px-6 text-left">Category</th>
                            <th className="py-3 px-6 text-left">Price ($)</th>
                            <th className="py-3 px-6 text-left">Stock</th>
                            <th className="py-3 px-6 text-center">Actions</th>
                        </tr>
                    </thead>
                    <tbody className="text-gray-700">
                        {products.map((product) => (
                            <tr key={product.id} className="border-b border-gray-200 hover:bg-gray-50">
                                <td className="py-3 px-6">{product.name}</td>
                                <td className="py-3 px-6">{product.category}</td>
                                <td className="py-3 px-6">${product.price.toFixed(2)}</td>
                                <td className="py-3 px-6">{product.stock}</td>
                                <td className="py-3 px-6 text-center space-x-2">
                                    <button
                                        onClick={() => handleEdit(product)}
                                        className="bg-yellow-500 text-white px-3 py-1 rounded hover:bg-yellow-600"
                                    >
                                        Edit
                                    </button>
                                    <button
                                        onClick={() => handleDelete(product.id)}
                                        className="bg-red-500 text-white px-3 py-1 rounded hover:bg-red-600"
                                    >
                                        Delete
                                    </button>
                                </td>
                            </tr>
                        ))}
                    </tbody>
                </table>
            </div>
        </div>
    );
}
