import { useState, useContext } from "react";
import { Link, useNavigate } from "react-router-dom";
import { FontAwesomeIcon } from "@fortawesome/react-fontawesome";
import {
    faUser,
    faShoppingCart,
    faStore,
    faBars,
    faX,
    faBox,
    faHeart,
    faGift,
    faTrophy,
    faMagnifyingGlass,
    faRightFromBracket,
} from "@fortawesome/free-solid-svg-icons";
import { AuthContext } from "../../context/AuthContext";
import axios from "axios";
import logo from "../../assets/images/logo_plain.png";

function Navbar() {
    const [isMobileMenuOpen, setMobileMenuOpen] = useState(false);
    const [searchQuery, setSearchQuery] = useState(""); // controlled input
    const { isLoggedIn, logout } = useContext(AuthContext);
    const navigate = useNavigate();

    const toggleMobileMenu = () => setMobileMenuOpen(!isMobileMenuOpen);

    const handleLogout = async () => {
        try {
            await axios.post(
                "http://localhost:9000/api/userlogout",
                { withCredentials: true }
            );
            logout();
            navigate("/userlogin");
        } catch (error) {
            console.error("Logout failed:", error.response?.data || error.message);
        }
    };

    return (
        <header className="relative bg-white shadow-md">
            <div className="flex justify-between items-center px-4 sm:px-8 py-3">
                {/* Logo */}
                <Link to="/" className="flex items-center gap-2">
                    <img className="w-10 h-10 object-contain" src={logo} alt="Esahakara" />
                    <div className="flex flex-col">
                        <span className="font-bold text-2xl bg-gradient-to-r from-orange-600 to-yellow-400 bg-clip-text text-transparent">
                            ESAHAKARA
                        </span>
                        <span className="font-bold text-sm bg-gradient-to-r from-orange-600 to-yellow-400 bg-clip-text text-transparent">
                            Products
                        </span>
                    </div>
                </Link>

                {/* Desktop Search */}
                <div className="hidden md:flex flex-1 max-w-[500px] mx-8 shadow-md rounded-lg overflow-hidden">
                    <input
                        className="flex-1 px-4 sm:px-8 pr-10 py-2 border border-gray-300 border-r-0 rounded-l-lg outline-none text-base"
                        type="text"
                        placeholder="Search for Products, Brands and More"
                        value={searchQuery}
                        onChange={(e) => setSearchQuery(e.target.value)}
                    />
                    <button className="px-4 sm:px-6 py-2 border border-gray-300 border-l-0 rounded-r-lg bg-orange-600 text-white font-bold cursor-pointer">
                        <FontAwesomeIcon icon={faMagnifyingGlass} />
                    </button>
                </div>

                {/* Desktop Menu */}
                <div className="hidden md:flex items-start gap-6">
                    {/* Account/Login Dropdown */}
                    <div className="relative group">
                        {!isLoggedIn ? (
                            <Link
                                to="userlogin"
                                className="bg-none border border-transparent text-gray-600 cursor-pointer text-base p-2.5 rounded-full flex items-start gap-2 transition-all duration-200 ease-in-out hover:bg-yellow-100 hover:text-orange-600 hover:border-orange-600"
                            >
                                <FontAwesomeIcon icon={faUser} /> Login
                            </Link>
                        ) : (
                            <button className="bg-none border border-transparent text-gray-600 cursor-pointer text-base p-2.5 rounded-full flex items-center gap-2 transition-all duration-200 ease-in-out hover:bg-yellow-100 hover:text-orange-600 hover:border-orange-600">
                                <FontAwesomeIcon icon={faUser} /> Account
                            </button>
                        )}

                        <div className="absolute top-12 left-0 w-60 bg-white text-gray-800 rounded-md shadow-lg opacity-0 group-hover:opacity-100 transition-opacity duration-200 pointer-events-none group-hover:pointer-events-auto z-50 hidden md:block">
                            {!isLoggedIn && (
                                <div className="flex justify-between p-3 text-sm font-medium">
                                    <span>New customer?</span>
                                    <Link
                                        to="/usersignup"
                                        className="text-orange-600 font-bold hover:underline"
                                    >
                                        Sign Up
                                    </Link>
                                </div>
                            )}
                            <hr />
                            <ul className="flex flex-col">
                                <li className="flex items-center gap-2 p-3 text-sm hover:bg-gray-100 cursor-pointer">
                                    <Link to="/userprofile" className="flex items-center gap-2">
                                        <FontAwesomeIcon icon={faUser} /> My Profile
                                    </Link>
                                </li>
                                <li className="flex items-center gap-2 p-3 text-sm hover:bg-gray-100 cursor-pointer">
                                    <Link to="/orders" className="flex items-center gap-2">
                                        <FontAwesomeIcon icon={faBox} /> Orders
                                    </Link>
                                </li>
                                <li className="flex items-center gap-2 p-3 text-sm hover:bg-gray-100 cursor-pointer">
                                    <Link to="/wishlist" className="flex items-center gap-2">
                                        <FontAwesomeIcon icon={faHeart} /> Wishlist
                                    </Link>
                                </li>
                                <li className="flex items-center gap-2 p-3 text-sm hover:bg-gray-100 cursor-pointer">
                                    <Link to="/rewards" className="flex items-center gap-2">
                                        <FontAwesomeIcon icon={faTrophy} /> Rewards
                                    </Link>
                                </li>
                                <li className="flex items-center gap-2 p-3 text-sm hover:bg-gray-100 cursor-pointer">
                                    <Link to="/gift-cards" className="flex items-center gap-2">
                                        <FontAwesomeIcon icon={faGift} /> Gift Cards
                                    </Link>
                                </li>
                                {isLoggedIn && (
                                    <li className="flex items-center gap-2 p-3 text-sm hover:bg-gray-100 cursor-pointer">
                                        <button
                                            onClick={handleLogout}
                                            className="flex items-start gap-2 w-full text-left"
                                        >
                                            <FontAwesomeIcon icon={faRightFromBracket} /> Logout
                                        </button>
                                    </li>
                                )}
                            </ul>
                        </div>
                    </div>

                    {/* Other Menu Buttons */}
                    <Link
                        to="/cart"
                        className="bg-none border border-transparent text-gray-600 cursor-pointer text-base p-2.5 rounded-full transition-all duration-200 ease-in-out hover:bg-yellow-100 hover:text-orange-600 hover:border-orange-600 flex items-start gap-2"
                    >
                        <FontAwesomeIcon icon={faShoppingCart} /> Cart
                    </Link>
                    <Link
                        to="/becomeaseller"
                        className="bg-none border border-transparent text-gray-600 cursor-pointer text-base p-2.5 rounded-full transition-all duration-200 ease-in-out hover:bg-yellow-100 hover:text-orange-600 hover:border-orange-600 flex items-start gap-2"
                    >
                        <FontAwesomeIcon icon={faStore} /> Become a Seller
                    </Link>
                </div>

                {/* Mobile Menu Toggle */}
                <div className="flex md:hidden items-center gap-4">
                    <button className="text-xl cursor-pointer" onClick={toggleMobileMenu}>
                        <FontAwesomeIcon icon={isMobileMenuOpen ? faX : faBars} />
                    </button>
                </div>
            </div>

            {/* Mobile Menu */}
            <div
                className={`md:hidden flex flex-col gap-4 px-4 pt-4 pb-4 bg-white shadow-md overflow-hidden transition-all duration-500 ease-in-out ${isMobileMenuOpen ? "max-h-screen" : "max-h-0"
                    }`}
                style={{ alignItems: "flex-start" }}
            >
                {!isLoggedIn ? (
                    <Link
                        to="/userlogin"
                        className="w-full text-left bg-none border border-transparent text-gray-600 cursor-pointer text-base p-2.5 rounded-full flex items-center gap-2 hover:bg-yellow-100 hover:text-orange-600 hover:border-orange-600 transition-all duration-200 ease-in-out"
                    >
                        <FontAwesomeIcon icon={faUser} /> Login
                    </Link>
                ) : (
                    <button
                        onClick={handleLogout}
                        className="w-full text-left bg-none border border-transparent text-gray-600 cursor-pointer text-base p-2.5 rounded-full flex items-center gap-2 hover:bg-yellow-100 hover:text-orange-600 hover:border-orange-600 transition-all duration-200 ease-in-out"
                    >
                        <FontAwesomeIcon icon={faRightFromBracket} /> Logout
                    </button>
                )}

                <Link
                    to="/cart"
                    className="w-full text-left bg-none border border-transparent text-gray-600 cursor-pointer text-base p-2.5 rounded-full flex items-center gap-2 hover:bg-yellow-100 hover:text-orange-600 hover:border-orange-600 transition-all duration-200 ease-in-out"
                >
                    <FontAwesomeIcon icon={faShoppingCart} /> Cart
                </Link>

                <Link
                    to="/becomeaseller"
                    className="w-full text-left bg-none border border-transparent text-gray-600 cursor-pointer text-base p-2.5 rounded-full flex items-center gap-2 hover:bg-yellow-100 hover:text-orange-600 hover:border-orange-600 transition-all duration-200 ease-in-out"
                >
                    <FontAwesomeIcon icon={faStore} /> Become a Seller
                </Link>
            </div>
        </header>
    );
}

export default Navbar;
